<?php
defined( 'ABSPATH' ) || exit;

class HRM_Setup {

    const MENU_SLUG          = 'hr-management';
    const EMPLOYEE_POST_TYPE = 'hrm_employee';

    public static function init() {
        add_action( 'init', [ __CLASS__, 'register_post_types' ] );
        add_action( 'admin_menu', [ __CLASS__, 'register_menu' ] );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_admin_assets' ] );
        add_action( 'admin_init', [ __CLASS__, 'restrict_admin_access' ] );
        add_action( 'after_setup_theme', [ __CLASS__, 'hide_admin_bar' ] );
        add_action( 'admin_init', [ __CLASS__, 'check_db_upgrade' ] );
    }

    public static function check_db_upgrade() {
        // Simple versioning to trigger dbDelta
        if ( get_option( 'hrm_plugin_db_version' ) !== '1.0.4' ) {
            self::activate_plugin();
            update_option( 'hrm_plugin_db_version', '1.0.4' );
        }
    }

    public static function register_post_types() {
        register_post_type( self::EMPLOYEE_POST_TYPE, [
            'labels' => [
                'name' => 'Employees',
                'singular_name' => 'Employee'
            ],
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => false,
            'supports' => [ 'title', 'custom-fields', 'author' ],
            'menu_icon' => 'dashicons-id'
        ]);
        
        // Register Custom Role
        add_role( 'hrm_employee', 'Employee', [
            'read' => true,
            'edit_posts' => false,
            'delete_posts' => false,
        ]);
    }

    public static function register_menu() {
        add_menu_page(
            'HR Management',
            'HR Management',
            'manage_options',
            self::MENU_SLUG,
            [ 'HRM_Dashboard', 'render_dashboard' ],
            'dashicons-groups',
            25
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Dashboard',
            'Dashboard',
            'manage_options',
            self::MENU_SLUG,
            [ 'HRM_Dashboard', 'render_dashboard' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Employees',
            'Employees',
            'manage_options',
            'hrm-employees',
            [ 'HRM_Employees', 'render_employees' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Attendance',
            'Attendance',
            'manage_options',
            'hrm-attendance',
            [ 'HRM_Attendance', 'render_attendance' ]
        );



        add_submenu_page(
            self::MENU_SLUG,
            'Daily Progress',
            'Daily Progress',
            'manage_options',
            'hrm-daily-progress',
            [ 'HRM_Tasks', 'render_daily_progress' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Assigned Tasks',
            'Assigned Tasks',
            'manage_options',
            'hrm-tasks',
            [ 'HRM_Tasks', 'render_tasks' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Leave Management',
            'Leave Management',
            'manage_options',
            'hrm-leave',
            [ 'HRM_Leave', 'render_admin_leave' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Reports',
            'Reports',
            'manage_options',
            'hrm-reports',
            [ 'HRM_Reports', 'render_reports' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Company Policies',
            'Company Policies',
            'manage_options',
            'hrm-policies',
            [ 'HRM_Policies', 'render_admin_policies' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Payroll',
            'Payroll',
            'manage_options',
            'hrm-payroll',
            [ 'HRM_Payroll', 'render_payroll' ]
        );

        add_submenu_page(
            self::MENU_SLUG,
            'Settings',
            'Settings',
            'manage_options',
            'hrm-settings',
            [ 'HRM_Dashboard', 'render_settings_page' ]
        );

        add_submenu_page(
            null,
            'Employee Profile',
            'Employee Profile',
            'manage_options',
            'hrm-employee-profile',
            [ 'HRM_Employees', 'render_employee_profile' ]
        );
    }

    public static function hide_admin_bar() {
        if ( ! current_user_can( 'manage_options' ) && ! is_admin() ) {
            show_admin_bar( false );
        }
    }

    public static function restrict_admin_access() {
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            return;
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_redirect( home_url( '/employee-portal/' ) );
            exit;
        }
    }

    public static function enqueue_admin_assets() {
        if ( ! defined('HRM_PLUGIN_URL') ) return;
        $plugin_url = HRM_PLUGIN_URL;
        $version = time(); 
        wp_enqueue_style( 'hrm-admin-css', $plugin_url . 'assets/css/hrm-admin.css', [], $version );
        wp_enqueue_style( 'hrm-icons-css', $plugin_url . 'assets/css/hrm-icons.css', ['hrm-admin-css'], $version );
        wp_enqueue_style( 'hrm-components-css', $plugin_url . 'assets/css/hrm-components.css', ['hrm-icons-css'], $version );
        wp_enqueue_style( 'hrm-ui-polish-css', $plugin_url . 'assets/css/hrm-ui-polish.css', ['hrm-components-css'], $version );
        wp_enqueue_style( 'hrm-responsive-fixes-css', $plugin_url . 'assets/css/hrm-responsive-fixes.css', ['hrm-ui-polish-css'], $version );
        wp_enqueue_style( 'hrm-responsive-edge-cases-css', $plugin_url . 'assets/css/hrm-responsive-edge-cases.css', ['hrm-responsive-fixes-css'], $version );
        wp_enqueue_script( 'hrm-admin-js', $plugin_url . 'assets/js/hrm-admin.js', ['jquery'], $version, true );
        wp_enqueue_script( 'hrm-ui-enhancer-js', $plugin_url . 'assets/js/hrm-ui-enhancer.js', ['jquery', 'hrm-admin-js'], $version, true );
        
        // Enqueue Reports assets on the reports page
        $screen = get_current_screen();
        if ( $screen && $screen->id === 'hr-management_page_hrm-reports' ) {
            wp_enqueue_style( 'hrm-reports-css', $plugin_url . 'assets/css/hrm-reports.css', [], $version );
            wp_enqueue_script( 'hrm-reports-js', $plugin_url . 'assets/js/hrm-reports.js', ['jquery'], $version, true );
        }
        
        // Enqueue Payroll module enhanced styles
        if ( $screen && $screen->id === 'hr-management_page_hrm-payroll' ) {
            wp_enqueue_style( 'hrm-payroll-css', $plugin_url . 'assets/css/hrm-payroll.css', ['hrm-admin-css'], $version );
        }
        
        wp_localize_script( 'hrm-admin-js', 'hrm_admin_params', [
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'hrm_ajax_nonce' )
        ]);
    }

    public static function enqueue_frontend_assets() {
        if ( ! defined('HRM_PLUGIN_URL') ) return;
        $plugin_url = HRM_PLUGIN_URL;
        wp_enqueue_style( 'dashicons' );
        wp_enqueue_style( 'hrm-frontend-css', $plugin_url . 'assets/css/hrm-frontend.css', [], time() );
        wp_enqueue_style( 'hrm-icons-css', $plugin_url . 'assets/css/hrm-icons.css', ['hrm-frontend-css'], time() );
        wp_enqueue_style( 'hrm-components-css', $plugin_url . 'assets/css/hrm-components.css', ['hrm-icons-css'], time() );
        wp_enqueue_style( 'hrm-ui-polish-css', $plugin_url . 'assets/css/hrm-ui-polish.css', ['hrm-components-css'], time() );
        wp_enqueue_style( 'hrm-responsive-fixes-css', $plugin_url . 'assets/css/hrm-responsive-fixes.css', ['hrm-ui-polish-css'], time() );
        wp_enqueue_style( 'hrm-responsive-edge-cases-css', $plugin_url . 'assets/css/hrm-responsive-edge-cases.css', ['hrm-responsive-fixes-css'], time() );
        
        wp_register_script( 'hrm-frontend-js', $plugin_url . 'assets/js/hrm-frontend.js', ['jquery'], time(), true );
        wp_localize_script( 'hrm-frontend-js', 'hrm_ajax', [
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'hrm_ajax_nonce' )
        ]);
        wp_enqueue_script( 'hrm-frontend-js' );
    }

    public static function activate_plugin() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Tables
        $tables = [
            "CREATE TABLE {$wpdb->prefix}hrm_attendance (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                date date NOT NULL,
                check_in datetime DEFAULT NULL,
                check_out datetime DEFAULT NULL,
                user_ip varchar(45) DEFAULT NULL,
                hours_worked float DEFAULT 0,
                remarks text DEFAULT NULL,
                status varchar(20) DEFAULT 'absent',
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY  (id)
            ) $charset_collate;",

            
            "CREATE TABLE {$wpdb->prefix}hrm_daily_progress (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                task_id bigint(20) UNSIGNED DEFAULT NULL,
                date date NOT NULL,
                hours_logged float DEFAULT 0,
                description text NOT NULL,
                status varchar(20) DEFAULT 'in_progress',
                admin_comment text DEFAULT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY  (id)
            ) $charset_collate;",
            
            "CREATE TABLE {$wpdb->prefix}hrm_tasks (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                assigned_to bigint(20) UNSIGNED NOT NULL,
                title varchar(255) NOT NULL,
                description text,
                due_date date,
                status varchar(20) DEFAULT 'pending',
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY  (id)
            ) $charset_collate;",
            
            "CREATE TABLE {$wpdb->prefix}hrm_notifications (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                type varchar(50) NOT NULL,
                message text NOT NULL,
                related_id bigint(20) UNSIGNED DEFAULT NULL,
                is_read tinyint(1) DEFAULT 0,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY  (id)
            ) $charset_collate;",
            
            "CREATE TABLE {$wpdb->prefix}hrm_leave (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                leave_type varchar(50) NOT NULL,
                start_date date NOT NULL,
                end_date date NOT NULL,
                reason text NOT NULL,
                status varchar(20) DEFAULT 'pending',
                admin_remarks text DEFAULT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY  (id),
                KEY employee_id (employee_id),
                KEY status (status)
            ) $charset_collate;",
            
            "CREATE TABLE {$wpdb->prefix}hrm_security_logs (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                event_type varchar(50) NOT NULL,
                message text NOT NULL,
                user_id bigint(20) UNSIGNED DEFAULT 0,
                ip_address varchar(45) NOT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
                PRIMARY KEY  (id)
            ) $charset_collate;",

            "CREATE TABLE {$wpdb->prefix}hrm_policies (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                title varchar(255) NOT NULL,
                category varchar(50) NOT NULL,
                description longtext NOT NULL,
                effective_date date NOT NULL,
                status varchar(20) DEFAULT 'draft',
                attachment_url varchar(500) DEFAULT NULL,
                attachment_id bigint(20) UNSIGNED DEFAULT NULL,
                created_by bigint(20) UNSIGNED NOT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY  (id),
                KEY status (status),
                KEY category (category)
            ) $charset_collate;",

            "CREATE TABLE {$wpdb->prefix}hrm_salary_structures (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                basic_salary decimal(15,2) DEFAULT 0,
                allowances text,
                deductions text,
                overtime_rate decimal(10,2) DEFAULT 0,
                tax_rules text,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY employee_id (employee_id)
            ) $charset_collate;",

            "CREATE TABLE {$wpdb->prefix}hrm_payroll (
                id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                employee_id bigint(20) UNSIGNED NOT NULL,
                payroll_month varchar(7) NOT NULL,
                basic_salary decimal(15,2) DEFAULT 0,
                total_allowances decimal(15,2) DEFAULT 0,
                total_deductions decimal(15,2) DEFAULT 0,
                total_bonuses decimal(15,2) DEFAULT 0,
                overtime_pay decimal(15,2) DEFAULT 0,
                attendance_penalty decimal(15,2) DEFAULT 0,
                tax_amount decimal(15,2) DEFAULT 0,
                net_salary decimal(15,2) DEFAULT 0,
                status varchar(20) DEFAULT 'draft',
                payslip_data longtext,
                created_by bigint(20) UNSIGNED NOT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY emp_month (employee_id, payroll_month)
            ) $charset_collate;"
        ];

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        foreach($tables as $sql) {
            dbDelta( $sql );
        }

        // Create Pages
        $pages = [
            'employee-login' => [ 'title' => 'Employee Login', 'content' => '[hrm_login]' ],
            'employee-portal' => [ 'title' => 'Employee Portal', 'content' => '[hrm_employee_portal]' ]
        ];

        foreach ( $pages as $slug => $page ) {
            if ( ! get_page_by_path( $slug ) ) {
                wp_insert_post([
                    'post_title'   => $page['title'],
                    'post_content' => $page['content'],
                    'post_status'  => 'publish',
                    'post_type'    => 'page',
                    'post_name'    => $slug
                ]);
            }
        }

        // Schedule Cron
        if ( ! wp_next_scheduled( 'hrm_daily_absent_check' ) ) {
            $time = strtotime( '23:59:00' );
            if ( $time < time() ) $time += DAY_IN_SECONDS;
            wp_schedule_event( $time, 'daily', 'hrm_daily_absent_check' );
        }
        
        // Hashing - we will assume the main file is handled by HRM_Security::init or similar
        // or just update it here for the current file context if we want to secure this file too.
        // But for the main file check, we should probably let the main file handle it or pass it.
        if ( defined('HRM_PLUGIN_FILE') ) {
            update_option( 'hrm_core_file_hash', hash_file( 'sha256', HRM_PLUGIN_FILE ) );
        }
    }
}
